<?php
require_once 'config.php';

/**
 * Gelen mesajları işle
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param array $mesaj Mesaj verisi
 */
function mesajIsleme($bot, $mesaj) {
    global $vt;
    
    $chat_id = $mesaj['chat']['id'];
    $text = $mesaj['text'] ?? '';
    $kullanici = $mesaj['from'];
    
    // Kullanıcıyı kaydet
    kullaniciKaydet($vt, $kullanici);
    
    // Komutları işle
    if (strpos($text, '/start') === 0) {
        // Referans kodu kontrolü
        $referans_kodu = substr($text, 7);
        if (!empty($referans_kodu)) {
            referansIsle($bot, $chat_id, $kullanici, $referans_kodu);
        } else {
            karsilamaGonder($bot, $chat_id);
        }
    } elseif ($text == '/yardim' || $text == '/help') {
        yardimGonder($bot, $chat_id);
    } elseif ($text == '/lidertablosu') {
        liderTablosuGonder($bot, $chat_id);
    } elseif ($text == '/istatistiklerim') {
        istatistikGonder($bot, $chat_id, $kullanici['id']);
    }
    // Sabit butonlar için mesaj işleme
    elseif ($text == '🔗 TIKLA HEMEN KAZANMAYA BAŞLA') {
        davetLinkiGonder($bot, $chat_id, $kullanici['id']);
    } elseif ($text == '📊 Lider Tablosu') {
        liderTablosuGonder($bot, $chat_id);
    } elseif ($text == '📈 İstatistiklerim') {
        istatistikGonder($bot, $chat_id, $kullanici['id']);
    } elseif ($text == '📜 Turnuva Kuralları') {
        turnuvaKurallariGonder($bot, $chat_id);
    } elseif ($text == '👤 Hemen Üye Ol!') {
        // Kayıt URL'sini gönder
        $bot->mesajGonder($chat_id, "Betplay'e üye olmak için aşağıdaki linki kullanabilirsiniz:\n\n" . KAYIT_URL);
    }
    
    // Telefon numarası paylaşımını işle
    if (isset($mesaj['contact']) && isset($mesaj['contact']['phone_number'])) {
        telefonKaydet($bot, $chat_id, $kullanici['id'], $mesaj['contact']['phone_number']);
    }
}

/**
 * Callback sorguları işle (buton tıklamaları)
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param array $callback_query Callback sorgu verisi
 */
function callbackIsleme($bot, $callback_query) {
    global $vt;
    
    $data = $callback_query['data'];
    $chat_id = $callback_query['message']['chat']['id'];
    $kullanici = $callback_query['from'];
    
    // Callback sorguyu yanıtla (yükleniyor durumunu kaldır)
    $bot->callbackYanitla($callback_query['id']);
    
    if ($data == 'davet_linki_al') {
        // Kullanıcının gerekli kanalları takip edip etmediğini kontrol et
        if (kanallariTakipEdiyor($bot, $kullanici['id'])) {
            // Telefon numarası sağlanmış mı kontrol et
            if (telefonuVar($vt, $kullanici['id'])) {
                davetLinkiGonder($bot, $chat_id, $kullanici['id']);
            } else {
                telefonIste($bot, $chat_id);
            }
        } else {
            kanalTakipIste($bot, $chat_id);
        }
    } elseif ($data == 'lider_tablosu') {
        liderTablosuGonder($bot, $chat_id);
    } elseif ($data == 'istatistiklerim') {
        istatistikGonder($bot, $chat_id, $kullanici['id']);
    } elseif ($data == 'kurallar') {
        turnuvaKurallariGonder($bot, $chat_id);
    } elseif ($data == 'hemen_uye_ol') {
        kayitLinkiGonder($bot, $chat_id);
    } elseif (strpos($data, 'takip_kontrol') === 0) {
        takipKontrolEt($bot, $chat_id, $kullanici['id']);
    }
}

/**
 * Kullanıcıyı veritabanına kaydet
 * 
 * @param mysqli $vt Veritabanı bağlantısı
 * @param array $kullanici Kullanıcı verisi
 */
function kullaniciKaydet($vt, $kullanici) {
    $telegram_id = $kullanici['id'];
    $kullanici_adi = $kullanici['username'] ?? '';
    $ad = $kullanici['first_name'] ?? '';
    $soyad = $kullanici['last_name'] ?? '';
    
    // Kullanıcının var olup olmadığını kontrol et
    $stmt = $vt->prepare("SELECT id FROM kullanicilar WHERE telegram_id = ?");
    $stmt->bind_param("i", $telegram_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows == 0) {
        // Benzersiz davet kodu oluştur
        $davet_kodu = davetKoduOlustur();
        $davet_linki = "t.me/" . substr(BOT_KULLANICI_ADI, 1) . "?start=" . $davet_kodu;
        
        // Yeni kullanıcı ekle
        $stmt = $vt->prepare("INSERT INTO kullanicilar (telegram_id, kullanici_adi, ad, soyad, davet_linki, davet_kodu, kayit_tarihi, aktif) VALUES (?, ?, ?, ?, ?, ?, NOW(), TRUE)");
        $stmt->bind_param("isssss", $telegram_id, $kullanici_adi, $ad, $soyad, $davet_linki, $davet_kodu);
        $stmt->execute();
    } else {
        // Mevcut kullanıcıyı güncelle
        $stmt = $vt->prepare("UPDATE kullanicilar SET kullanici_adi = ?, ad = ?, soyad = ?, aktif = TRUE WHERE telegram_id = ?");
        $stmt->bind_param("sssi", $kullanici_adi, $ad, $soyad, $telegram_id);
        $stmt->execute();
    }
}

/**
 * Benzersiz bir davet kodu oluştur
 * 
 * @return string Davet kodu
 */
function davetKoduOlustur() {
    return rand(1000000000, 9999999999);
}

/**
 * Zorunlu kanalları veritabanından al
 * 
 * @return array Zorunlu kanallar listesi
 */
function zorunluKanallariGetir() {
    global $vt;
    
    $kanallar = [];
    $sorgu = $vt->query("SELECT kanal_kullanici_adi FROM kanallar WHERE zorunlu = 1");
    
    if ($sorgu && $sorgu->num_rows > 0) {
        while ($row = $sorgu->fetch_assoc()) {
            $kanallar[] = $row['kanal_kullanici_adi'];
        }
    }
    
    return $kanallar;
}

/**
 * Kullanıcının gerekli kanalları takip edip etmediğini kontrol et
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param int $user_id Kullanıcı ID
 * @return bool Tüm kanalları takip ediyor mu
 */
function kanallariTakipEdiyor($bot, $user_id) {
    $zorunlu_kanallar = zorunluKanallariGetir();
    
    foreach ($zorunlu_kanallar as $kanal) {
        $sonuc = $bot->sohbetUyesiAl($kanal, $user_id);
        
        if (!$sonuc || !$sonuc['ok'] || $sonuc['result']['status'] == 'left' || $sonuc['result']['status'] == 'kicked') {
            return false;
        }
    }
    
    return true;
}

/**
 * Kanal aboneliği iste
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param int $chat_id Sohbet ID
 */
function kanalTakipIste($bot, $chat_id) {
    $zorunlu_kanallar = zorunluKanallariGetir();
    $mesaj = "Davet linkini alabilmek için lütfen aşağıdaki kanalları takip et:";
    
    $klavye = [];
    foreach ($zorunlu_kanallar as $kanal) {
        $mesaj .= "\n" . $kanal;
        $klavye[] = [['text' => $kanal, 'url' => 'https://t.me/' . str_replace('@', '', $kanal)]];
    }
    
    $klavye[] = [['text' => '✅ Takip Ettim', 'callback_data' => 'takip_kontrol']];
    
    $bot->mesajGonder($chat_id, $mesaj, '', false, null, ['inline_keyboard' => $klavye]);
}

/**
 * Kullanıcının telefon numarası olup olmadığını kontrol et
 * 
 * @param mysqli $vt Veritabanı bağlantısı
 * @param int $telegram_id Telegram ID
 * @return bool Telefon numarası var mı veya gerekli değil mi
 */
function telefonuVar($vt, $telegram_id) {
    // Telefon zorunlu mu kontrol et
    $telefon_zorunlu = ayarAl('TELEFON_ZORUNLU', '1');
    
    // Telefon zorunlu değilse, true döndür (telefon numarası gerekli değil)
    if ($telefon_zorunlu == '0') {
        return true;
    }
    
    // Telefon zorunlu ise, kullanıcının telefon numarası var mı kontrol et
    $stmt = $vt->prepare("SELECT telefon FROM kullanicilar WHERE telegram_id = ?");
    $stmt->bind_param("i", $telegram_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($row = $result->fetch_assoc()) {
        return !empty($row['telefon']);
    }
    
    return false;
}

/**
 * Telefon numarası iste
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param int $chat_id Sohbet ID
 */
function telefonIste($bot, $chat_id) {
    // Telefon numarası paylaşma butonu
    $keyboard = [
        [
            [
                'text' => '📱 Telefon Numaramı Paylaş',
                'request_contact' => true
            ]
        ]
    ];
    
    $reply_markup = [
        'keyboard' => $keyboard,
        'resize_keyboard' => true,
        'one_time_keyboard' => true
    ];
    
    $mesaj = "🎁 *Etkinliğe Katılın ve Kazanmaya Başlayın!*\n\n";
    $mesaj .= "Etkinliğe katılmak ve özel davet linkinizi almak için lütfen aşağıdaki butondan telefon numaranızı paylaşın.\n\n";
    $mesaj .= "Telefon numaranız sadece etkinlik kapsamında kullanılacak ve üçüncü taraflarla paylaşılmayacaktır.";
    
    // Mesajı gönder ve önceki mesajı sil
    $bot->mesajGonder($chat_id, $mesaj, 'Markdown', false, null, $reply_markup);
}

/**
 * Kullanıcının telefon numarasını kaydet
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param int $chat_id Sohbet ID
 * @param int $telegram_id Telegram ID
 * @param string $telefon Telefon numarası
 */
function telefonKaydet($bot, $chat_id, $telegram_id, $telefon) {
    global $vt;
    
    $stmt = $vt->prepare("UPDATE kullanicilar SET telefon = ? WHERE telegram_id = ?");
    $stmt->bind_param("si", $telefon, $telegram_id);
    $stmt->execute();
    
    if ($stmt->affected_rows > 0) {
        $bot->mesajGonder($chat_id, "Telefon numaranız başarıyla kaydedildi! Artık devam edebilirsiniz.");
        davetLinkiGonder($bot, $chat_id, $telegram_id);
    } else {
        $bot->mesajGonder($chat_id, "⚠️ Telefon numaranız kaydedilirken bir hata oluştu. Lütfen tekrar deneyin.");
    }
}

/**
 * Karşılama mesajı gönder
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param int $chat_id Sohbet ID
 */
function karsilamaGonder($bot, $chat_id) {
    global $vt;
    
    // Kullanıcı ID'sini al
    $user_id = $chat_id; // Özel sohbetlerde chat_id = user_id
    
    // Bekleyen bildirimleri kontrol et ve gönder
    bekleyenBildirimleriGonder($bot, $user_id);
    
    // Etkinlik aktif mi kontrol et
    if (!etkinlikAktifMi()) {
        $bot->mesajGonder($chat_id, "⚠️ Şu anda aktif bir etkinlik bulunmamaktadır. Lütfen daha sonra tekrar deneyin.");
        return;
    }
    
    // Karşılama mesajı
    $mesaj = "🏆 *BETPLAY TELEGRAM DAVET TURNUVASI*\n\n";
    $mesaj .= "Bu bot ne yapabilir?\n\n";
    $mesaj .= "⚡️ Arkadaşlarını davet et, ödüller kazan!\n\n";
    $mesaj .= "Davet linkini almak için 'TIKLA HEMEN KAZANMAYA BAŞLA' butonuna tıklayın.\n\n";
    $mesaj .= "Lütfen aşağıdaki butonlardan seçiminizi yapınız:";
    
    // Sabit butonlar için ReplyKeyboardMarkup kullan
    $keyboard = [
        ['🔗 TIKLA HEMEN KAZANMAYA BAŞLA'],
        ['📊 Lider Tablosu', '📈 İstatistiklerim'],
        ['📜 Turnuva Kuralları']
    ];
    
    $reply_markup = [
        'keyboard' => $keyboard,
        'resize_keyboard' => true,
        'one_time_keyboard' => false,
        'persistent' => true
    ];
    
    // Tek bir mesaj gönder
    $bot->mesajGonder($chat_id, $mesaj, 'Markdown', false, null, $reply_markup);
    
    // Kayıt URL butonunu ekle (web_app özelliği için)
    // Bunu asenkron olarak yapabiliriz, kullanıcının beklemesine gerek yok
    $kayit_url = ayarAl('KAYIT_URL', 'https://betplay.com/kayit');
    $buton_metni = ayarAl('KAYIT_BUTON_METNI', 'BetPlay\'e Giriş yap.');
    
    // Menü butonunu ekle - bu işlemi arka planda yapabiliriz
    @$bot->request('setChatMenuButton', [
        'chat_id' => $chat_id,
        'menu_button' => json_encode([
            'type' => 'web_app',
            'text' => $buton_metni,
            'web_app' => [
                'url' => $kayit_url
            ]
        ])
    ]);
}

/**
 * Kayıt linkini gönder
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param int $chat_id Sohbet ID
 */
function kayitLinkiGonder($bot, $chat_id) {
    $klavye = [
        [['text' => 'Mobil Uygulama', 'callback_data' => 'mobil_uygulama']],
        [['text' => 'Güncel Giriş', 'callback_data' => 'guncel_giris']],
        [['text' => '🎰 EGT Jackpot Promosyonu 🎰', 'callback_data' => 'egt_promo']],
        [['text' => 'Betplay Çark', 'callback_data' => 'betplay_cark']]
    ];
    
    $mesaj = "Hemen Üye Ol!";
    
    $bot->mesajGonder($chat_id, $mesaj, '', true, null, ['inline_keyboard' => $klavye]);
    
    // Kayıt URL'sini veya web uygulamasını aç
    $bot->mesajGonder($chat_id, "Kayıt olmak için: " . KAYIT_URL);
}

/**
 * Abonelikleri kontrol et
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param int $chat_id Sohbet ID
 * @param int $user_id Kullanıcı ID
 */
function takipKontrolEt($bot, $chat_id, $user_id) {
    global $vt;
    
    if (kanallariTakipEdiyor($bot, $user_id)) {
        if (telefonuVar($vt, $user_id)) {
            davetLinkiGonder($bot, $chat_id, $user_id);
        } else {
            telefonIste($bot, $chat_id);
        }
    } else {
        $bot->mesajGonder($chat_id, "⚠️ Henüz tüm kanalları takip etmediniz. Lütfen tüm kanalları takip edip tekrar deneyin.");
    }
}

/**
 * Etkinliğin aktif olup olmadığını kontrol et
 * 
 * @return bool Etkinlik aktif mi
 */
function etkinlikAktifMi() {
    global $vt;
    
    // Etkinlik tarihlerini al
    $baslangic_sorgu = $vt->query("SELECT deger FROM ayarlar WHERE anahtar = 'TURNUVA_BASLANGIC'");
    $bitis_sorgu = $vt->query("SELECT deger FROM ayarlar WHERE anahtar = 'TURNUVA_BITIS'");
    
    $baslangic = $baslangic_sorgu->fetch_assoc()['deger'];
    $bitis = $bitis_sorgu->fetch_assoc()['deger'];
    $simdi = date('Y-m-d H:i:s');
    
    // Şu anki zaman, başlangıç ve bitiş tarihleri arasında mı kontrol et
    return ($simdi >= $baslangic && $simdi <= $bitis);
}

/**
 * Bekleyen bildirimleri gönder
 * 
 * @param BotAPI $bot Bot API nesnesi
 * @param int $user_id Kullanıcı ID
 */
function bekleyenBildirimleriGonder($bot, $user_id) {
    global $vt;
    
    // Kullanıcının bekleyen bildirimlerini al
    $bildirim_sorgu = $vt->prepare("
        SELECT id, mesaj 
        FROM bildirimler 
        WHERE telegram_id = ? AND gonderildi = 0
        ORDER BY olusturma_tarihi ASC
    ");
    $bildirim_sorgu->bind_param("i", $user_id);
    $bildirim_sorgu->execute();
    $bildirim_sonuc = $bildirim_sorgu->get_result();
    
    // Bildirimleri gönder
    while ($bildirim = $bildirim_sonuc->fetch_assoc()) {
        try {
            // Bildirimi gönder
            $bot->mesajGonder($user_id, $bildirim['mesaj'], 'Markdown');
            
            // Bildirimi gönderildi olarak işaretle
            $guncelle_sorgu = $vt->prepare("
                UPDATE bildirimler 
                SET gonderildi = 1, gonderim_tarihi = NOW() 
                WHERE id = ?
            ");
            $guncelle_sorgu->bind_param("i", $bildirim['id']);
            $guncelle_sorgu->execute();
            
            // Çok fazla mesaj göndermeyi önlemek için kısa bir bekleme
            usleep(500000); // 0.5 saniye bekle
        } catch (Exception $e) {
            // Hata durumunda loglama yapılabilir
            error_log("Bildirim gönderme hatası: " . $e->getMessage());
        }
    }
}

/**
 * Rastgele string oluştur
 * 
 * @param int $length Uzunluk
 * @return string Rastgele string
 */
function generateRandomString($length = 8) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, $charactersLength - 1)];
    }
    return $randomString;
}
?>
