<?php
require_once 'config.php';
require_once 'BotAPI.php';

/**
 * Tüm aktif kullanıcılara bildirim gönder
 * 
 * @param string $mesaj Gönderilecek mesaj
 * @param string $format Mesaj formatı (markdown, html, vb.)
 * @return array Başarılı ve başarısız gönderim sayıları
 */
function tumKullanicilara($mesaj, $format = 'markdown') {
    global $vt;
    $bot = new BotAPI(BOT_TOKEN);
    
    $sonuc = [
        'basarili' => 0,
        'basarisiz' => 0
    ];
    
    // Aktif kullanıcıları al
    $sorgu = $vt->query("SELECT telegram_id FROM kullanicilar WHERE aktif = 1");
    
    while ($kullanici = $sorgu->fetch_assoc()) {
        try {
            $gonderim = $bot->mesajGonder($kullanici['telegram_id'], $mesaj, $format);
            if ($gonderim) {
                $sonuc['basarili']++;
            } else {
                $sonuc['basarisiz']++;
            }
            // Telegram API limitlerini aşmamak için kısa bir bekleme
            usleep(50000); // 0.05 saniye bekle
        } catch (Exception $e) {
            $sonuc['basarisiz']++;
            // Hata durumunda kullanıcıyı pasif yap
            $vt->query("UPDATE kullanicilar SET aktif = 0 WHERE telegram_id = " . $kullanici['telegram_id']);
        }
    }
    
    return $sonuc;
}

/**
 * Belirli kriterlere göre kullanıcılara bildirim gönder
 * 
 * @param string $mesaj Gönderilecek mesaj
 * @param array $kriterler SQL WHERE koşulları için dizi
 * @param string $format Mesaj formatı (markdown, html, vb.)
 * @return array Başarılı ve başarısız gönderim sayıları
 */
function filtreliKullanicilara($mesaj, $kriterler = [], $format = 'markdown') {
    global $vt;
    $bot = new BotAPI(BOT_TOKEN);
    
    $sonuc = [
        'basarili' => 0,
        'basarisiz' => 0
    ];
    
    // WHERE koşullarını oluştur
    $where = "WHERE aktif = 1";
    foreach ($kriterler as $alan => $deger) {
        $where .= " AND $alan = '" . $vt->real_escape_string($deger) . "'";
    }
    
    // Filtrelenmiş kullanıcıları al
    $sorgu = $vt->query("SELECT telegram_id FROM kullanicilar $where");
    
    while ($kullanici = $sorgu->fetch_assoc()) {
        try {
            $gonderim = $bot->mesajGonder($kullanici['telegram_id'], $mesaj, $format);
            if ($gonderim) {
                $sonuc['basarili']++;
            } else {
                $sonuc['basarisiz']++;
            }
            // Telegram API limitlerini aşmamak için kısa bir bekleme
            usleep(50000); // 0.05 saniye bekle
        } catch (Exception $e) {
            $sonuc['basarisiz']++;
        }
    }
    
    return $sonuc;
}

/**
 * En çok davet yapan kullanıcılara bildirim gönder
 * 
 * @param string $mesaj Gönderilecek mesaj
 * @param int $limit Kaç kullanıcıya gönderileceği
 * @param string $format Mesaj formatı (markdown, html, vb.)
 * @return array Başarılı ve başarısız gönderim sayıları
 */
function enCokDavetEdenlere($mesaj, $limit = 10, $format = 'markdown') {
    global $vt;
    $bot = new BotAPI(BOT_TOKEN);
    
    $sonuc = [
        'basarili' => 0,
        'basarisiz' => 0
    ];
    
    // En çok davet yapan kullanıcıları al
    $sorgu = $vt->query("
        SELECT 
            k.telegram_id,
            COUNT(d.id) as davet_sayisi
        FROM 
            kullanicilar k
        LEFT JOIN 
            davetler d ON k.telegram_id = d.davet_eden_id
        WHERE 
            k.aktif = 1
        GROUP BY 
            k.telegram_id
        ORDER BY 
            davet_sayisi DESC
        LIMIT $limit
    ");
    
    while ($kullanici = $sorgu->fetch_assoc()) {
        try {
            $gonderim = $bot->mesajGonder($kullanici['telegram_id'], $mesaj, $format);
            if ($gonderim) {
                $sonuc['basarili']++;
            } else {
                $sonuc['basarisiz']++;
            }
            // Telegram API limitlerini aşmamak için kısa bir bekleme
            usleep(50000); // 0.05 saniye bekle
        } catch (Exception $e) {
            $sonuc['basarisiz']++;
        }
    }
    
    return $sonuc;
}

/**
 * Minimum davet sayısını aşan kullanıcılara bildirim gönder
 * 
 * @param string $mesaj Gönderilecek mesaj
 * @param int $min_davet Minimum davet sayısı (varsayılan olarak config'den alınır)
 * @param string $format Mesaj formatı (markdown, html, vb.)
 * @return array Başarılı ve başarısız gönderim sayıları
 */
function minDavetAsanKullanicilara($mesaj, $min_davet = null, $format = 'markdown') {
    global $vt;
    $bot = new BotAPI(BOT_TOKEN);
    
    // Eğer minimum davet sayısı belirtilmemişse, config'den al
    if ($min_davet === null) {
        $min_davet = minDavetSayisiAl();
    }
    
    $sonuc = [
        'basarili' => 0,
        'basarisiz' => 0
    ];
    
    // Minimum davet sayısını aşan kullanıcıları al
    $sorgu = $vt->query("
        SELECT 
            k.telegram_id,
            COUNT(d.id) as davet_sayisi
        FROM 
            kullanicilar k
        JOIN 
            davetler d ON k.telegram_id = d.davet_eden_id
        WHERE 
            k.aktif = 1
        GROUP BY 
            k.telegram_id
        HAVING 
            davet_sayisi >= $min_davet
    ");
    
    while ($kullanici = $sorgu->fetch_assoc()) {
        try {
            $gonderim = $bot->mesajGonder($kullanici['telegram_id'], $mesaj, $format);
            if ($gonderim) {
                $sonuc['basarili']++;
            } else {
                $sonuc['basarisiz']++;
            }
            // Telegram API limitlerini aşmamak için kısa bir bekleme
            usleep(50000); // 0.05 saniye bekle
        } catch (Exception $e) {
            $sonuc['basarisiz']++;
        }
    }
    
    return $sonuc;
}

/**
 * Etkinlik başladığında tüm kullanıcılara bildirim gönder
 * 
 * @param object $vt Veritabanı bağlantısı
 * @return array Başarılı ve başarısız gönderim sayıları
 */
function etkinlikBasladiBildirimiGonder($vt) {
    // Ayarları veritabanından al
    $ayarlar = [];
    $ayarlar_sorgu = $vt->query("SELECT anahtar, deger FROM ayarlar");
    if ($ayarlar_sorgu) {
        while ($row = $ayarlar_sorgu->fetch_assoc()) {
            $ayarlar[$row['anahtar']] = $row['deger'];
        }
    }
    
    // Ödül bilgilerini veritabanından al
    $min_davet_sayisi = isset($ayarlar['MIN_DAVET_SAYISI']) ? $ayarlar['MIN_DAVET_SAYISI'] : 10;
    $freebet_miktar = isset($ayarlar['FREEBET_MIKTARI']) ? $ayarlar['FREEBET_MIKTARI'] : 200;
    $ekstra_odul_miktari = isset($ayarlar['EKSTRA_ODUL_MIKTARI']) ? $ayarlar['EKSTRA_ODUL_MIKTARI'] : 100;
    
    // Üst ödülleri JSON'dan çöz
    $ust_oduller_json = isset($ayarlar['UST_ODULLER']) ? $ayarlar['UST_ODULLER'] : '[{"sira":1,"odul":150000},{"sira":2,"odul":75000},{"sira":3,"odul":50000},{"sira":4,"odul":25000},{"sira":5,"odul":10000}]';
    $ust_oduller = json_decode($ust_oduller_json, true);
    
    // Toplam üst ödül miktarını hesapla
    $toplam_ust_odul = 0;
    foreach ($ust_oduller as $odul) {
        $toplam_ust_odul += $odul['odul'];
    }
    $ust_odul_sayisi = count($ust_oduller);
    
    // Üst ödüllerin detaylı listesini oluştur
    $ust_odul_detay = '';
    foreach ($ust_oduller as $odul) {
        if (isset($odul['sira']) && isset($odul['odul'])) {
            $emoji = '';
            switch ($odul['sira']) {
                case 1: $emoji = '🥇'; break;
                case 2: $emoji = '🥈'; break;
                case 3: $emoji = '🥉'; break;
                default: $emoji = '🏅'; break;
            }
            $ust_odul_detay .= "$emoji " . $odul['sira'] . ". sıra: " . number_format($odul['odul'], 0, ',', '.') . " TL\n";
        }
    }
    
    // Turnuva bitiş tarihini al
    $turnuva_bitis = isset($ayarlar['TURNUVA_BITIS']) ? $ayarlar['TURNUVA_BITIS'] : date('Y-m-d H:i:s', strtotime('+10 days'));
    $bitis_tarihi = date('d.m.Y', strtotime($turnuva_bitis));
    
    // Bildirim mesajını oluştur
    $mesaj = "🎮 *BETPLAY DAVET TURNUVASI BAŞLADI!* 🎮\n\n";
    $mesaj .= "🔥 Arkadaşlarını davet et, büyük ödüller kazan!\n\n";
    $mesaj .= "💎 *ÖDÜLLER:*\n";
    $mesaj .= "- En çok davet eden ilk " . $ust_odul_sayisi . " kişiye toplam " . number_format($toplam_ust_odul, 0, ',', '.') . " TL nakit!\n";
    $mesaj .= $ust_odul_detay . "\n";
    $mesaj .= "- Minimum " . $min_davet_sayisi . " arkadaş daveti = " . $freebet_miktar . " TL Freebet\n";
    $mesaj .= "- Her ekstra davet için +" . $ekstra_odul_miktari . " TL bonus!\n\n";
    $mesaj .= "🏆 Turnuva " . $bitis_tarihi . " tarihine kadar devam edecektir.\n\n";
    $mesaj .= "📱 Hemen davet linkini almak için bota yazın ve arkadaşlarınızı davet etmeye başlayın!\n";
    $mesaj .= "👉 /davet komutunu kullanarak davet linkinizi alabilirsiniz.";
    
    // Tüm kullanıcılara bildirim gönder
    return tumKullanicilara($mesaj);
}

/**
 * Etkinlik durdurulduğunda tüm kullanıcılara bildirim gönder
 * 
 * @param object $vt Veritabanı bağlantısı
 * @return array Başarılı ve başarısız gönderim sayıları
 */
function etkinlikDurdurulduBildirimiGonder($vt) {
    // Bildirim mesajını oluştur
    $mesaj = "⚠️ *BETPLAY DAVET TURNUVASI SONA ERDİ* ⚠️\n\n";
    $mesaj .= "Değerli katılımcılarımız,\n\n";
    $mesaj .= "Davet turnuvamız sona ermiştir. Kazananlar yakında belirlenecek ve ödüller dağıtılacaktır.\n\n";
    $mesaj .= "Katılımınız için teşekkür ederiz! Yeni turnuvalarımız için bizi takip etmeye devam edin.";
    
    // Tüm kullanıcılara bildirim gönder
    return tumKullanicilara($mesaj);
}

// Örnek kullanım:
// $sonuc = tumKullanicilara("🎉 *Yeni Turnuva Duyurusu*\n\nYeni turnuvamız başladı! Hemen arkadaşlarını davet et ve ödüller kazan!");
// echo "Başarılı: " . $sonuc['basarili'] . ", Başarısız: " . $sonuc['basarisiz'];

// Örnek kullanım:
// $sonuc = etkinlikBasladiBildirimiGonder($vt);
// echo "Başarılı: " . $sonuc['basarili'] . ", Başarısız: " . $sonuc['basarisiz'];

// Örnek kullanım:
// $sonuc = etkinlikDurdurulduBildirimiGonder($vt);
// echo "Başarılı: " . $sonuc['basarili'] . ", Başarısız: " . $sonuc['basarisiz'];
?>
