<?php
/**
 * Telegram Bot API için yardımcı sınıf
 */
class BotAPI {
    private $token;
    private $apiUrl = 'https://api.telegram.org/bot';
    
    /**
     * Yapılandırıcı
     * 
     * @param string $token Bot token
     */
    public function __construct($token) {
        $this->token = $token;
        $this->apiUrl .= $token . '/';
    }
    
    /**
     * Bir sohbete mesaj gönder
     * 
     * @param int|string $chat_id Sohbet ID
     * @param string $text Mesaj metni
     * @param string $parse_mode Biçimlendirme modu (markdown, html)
     * @param bool $disable_web_page_preview Web sayfası önizlemesini devre dışı bırak
     * @param int|null $reply_to_message_id Yanıtlanacak mesaj ID
     * @param array|null $reply_markup Özel klavye
     * @return array API yanıtı
     */
    public function mesajGonder($chat_id, $text, $parse_mode = '', $disable_web_page_preview = false, $reply_to_message_id = null, $reply_markup = null) {
        $params = [
            'chat_id' => $chat_id,
            'text' => $text,
            'disable_web_page_preview' => $disable_web_page_preview
        ];
        
        if (!empty($parse_mode)) {
            $params['parse_mode'] = $parse_mode;
        }
        
        if (!is_null($reply_to_message_id)) {
            $params['reply_to_message_id'] = $reply_to_message_id;
        }
        
        if (!is_null($reply_markup)) {
            $params['reply_markup'] = json_encode($reply_markup);
        }
        
        return $this->request('sendMessage', $params);
    }
    
    /**
     * Callback sorgusu yanıtla
     * 
     * @param string $callback_query_id Callback sorgu ID
     * @param string $text Metin
     * @param bool $show_alert Uyarı göster
     * @return array API yanıtı
     */
    public function callbackYanitla($callback_query_id, $text = '', $show_alert = false) {
        $params = [
            'callback_query_id' => $callback_query_id,
            'text' => $text,
            'show_alert' => $show_alert
        ];
        
        return $this->request('answerCallbackQuery', $params);
    }
    
    /**
     * Sohbet katılma isteğini onayla
     * 
     * @param string $chat_id Sohbet ID
     * @param int $user_id Kullanıcı ID
     * @return array API yanıtı
     */
    public function katilmaIstegiOnayla($chat_id, $user_id) {
        $params = [
            'chat_id' => $chat_id,
            'user_id' => $user_id
        ];
        
        return $this->request('approveChatJoinRequest', $params);
    }
    
    /**
     * Sohbet davet bağlantısı oluştur
     * 
     * @param string $chat_id Sohbet ID
     * @param string $name İsim
     * @param int|null $expire_date Son kullanma tarihi
     * @param int|null $member_limit Üye sınırı
     * @param bool $creates_join_request Katılma isteği oluştur
     * @return array API yanıtı
     */
    public function davetBaglantiOlustur($chat_id, $name = '', $expire_date = null, $member_limit = null, $creates_join_request = true) {
        $params = [
            'chat_id' => $chat_id,
            'creates_join_request' => $creates_join_request
        ];
        
        if (!empty($name)) {
            $params['name'] = $name;
        }
        
        if (!is_null($expire_date)) {
            $params['expire_date'] = $expire_date;
        }
        
        if (!is_null($member_limit)) {
            $params['member_limit'] = $member_limit;
        }
        
        return $this->request('createChatInviteLink', $params);
    }
    
    /**
     * Sohbet üyesini al
     * 
     * @param string $chat_id Sohbet ID
     * @param int $user_id Kullanıcı ID
     * @return array API yanıtı
     */
    public function sohbetUyesiAl($chat_id, $user_id) {
        $params = [
            'chat_id' => $chat_id,
            'user_id' => $user_id
        ];
        
        return $this->request('getChatMember', $params);
    }
    
    /**
     * Güncellemeleri al (polling için)
     * 
     * @param int $offset Başlangıç offset
     * @param int $limit Maksimum güncelleme sayısı
     * @param int $timeout Zaman aşımı süresi
     * @param array $allowed_updates İzin verilen güncelleme türleri
     * @return array API yanıtı
     */
    public function guncellemeleriAl($offset = 0, $limit = 100, $timeout = 0, $allowed_updates = []) {
        $params = [
            'offset' => $offset,
            'limit' => $limit,
            'timeout' => $timeout
        ];
        
        if (!empty($allowed_updates)) {
            $params['allowed_updates'] = json_encode($allowed_updates);
        }
        
        return $this->request('getUpdates', $params);
    }
    
    /**
     * Bot profilini güncelle
     * 
     * @param string $isim Bot ismi
     * @return array|false
     */
    public function botIsmiGuncelle($isim) {
        return $this->request('setMyName', [
            'name' => $isim
        ]);
    }
    
    /**
     * Bot açıklamasını güncelle
     * 
     * @param string $aciklama Bot açıklaması
     * @return array|false
     */
    public function botAciklamasiGuncelle($aciklama) {
        return $this->request('setMyDescription', [
            'description' => $aciklama
        ]);
    }
    
    /**
     * Bot kısa açıklamasını güncelle
     * 
     * @param string $kisa_aciklama Bot kısa açıklaması
     * @return array|false
     */
    public function botKisaAciklamasiGuncelle($kisa_aciklama) {
        return $this->request('setMyShortDescription', [
            'short_description' => $kisa_aciklama
        ]);
    }
    
    /**
     * Bot komutlarını güncelle
     * 
     * @param array $komutlar Komut listesi [['command' => 'start', 'description' => 'Botu başlat']]
     * @return array API yanıtı
     */
    public function botKomutlariniGuncelle($komutlar = []) {
        return $this->request('setMyCommands', [
            'commands' => json_encode($komutlar)
        ]);
    }
    
    /**
     * Bot menü butonlarını güncelle
     * 
     * @param array $butonlar Butonlar
     * @return array API yanıtı
     */
    public function botMenuButonlariniGuncelle($butonlar = []) {
        return $this->request('setMyCommands', [
            'commands' => json_encode($butonlar),
            'scope' => json_encode(['type' => 'default']),
            'language_code' => ''
        ]);
    }
    
    /**
     * Kayıt URL butonunu ekle
     * 
     * @param string $url Kayıt URL'si
     * @param string $buton_metni Buton metni
     * @return array API yanıtı
     */
    public function kayitUrlButonuEkle($url, $buton_metni = 'BetPlay\'e Giriş yap.') {
        $butonlar = [
            [
                'command' => 'betplay',
                'description' => 'BetPlay\'e giriş yap'
            ],
            [
                'command' => 'start',
                'description' => 'Botu başlat'
            ],
            [
                'command' => 'davet',
                'description' => 'Davet linkini al'
            ],
            [
                'command' => 'istatistiklerim',
                'description' => 'İstatistiklerini gör'
            ],
            [
                'command' => 'kurallar',
                'description' => 'Turnuva kurallarını gör'
            ]
        ];
        
        // Komutları güncelle
        $this->botKomutlariniGuncelle($butonlar);
        
        // Önce mevcut menü butonunu temizle
        $clearResult = $this->request('deleteMyCommands', [
            'scope' => json_encode(['type' => 'default'])
        ]);
        
        // Önbelleği temizlemek için önce boş bir buton ayarla
        $clearMenuResult = $this->request('setChatMenuButton', [
            'menu_button' => json_encode(['type' => 'default'])
        ]);
        
        // Kısa bir bekleme süresi
        usleep(500000); // 0.5 saniye bekle
        
        // Menü butonunu ekle - Önce global olarak ayarla
        $menuResult = $this->request('setChatMenuButton', [
            'menu_button' => json_encode([
                'type' => 'web_app',
                'text' => $buton_metni,
                'web_app' => [
                    'url' => $url
                ]
            ])
        ]);
        
        // Ayrıca özel olarak da ayarla
        $privateResult = $this->request('setChatMenuButton', [
            'chat_id' => null, // null = özel sohbetler için
            'menu_button' => json_encode([
                'type' => 'web_app',
                'text' => $buton_metni,
                'web_app' => [
                    'url' => $url
                ]
            ])
        ]);
        
        // Kullanıcılar için de ayarla (bazı durumlarda gerekli olabilir)
        $userScopeResult = $this->request('setChatMenuButton', [
            'scope' => json_encode(['type' => 'all_private_chats']),
            'menu_button' => json_encode([
                'type' => 'web_app',
                'text' => $buton_metni,
                'web_app' => [
                    'url' => $url
                ]
            ])
        ]);
        
        // Her iki sonucu da kontrol et
        if ($menuResult && $menuResult['ok']) {
            return $menuResult;
        } else if ($privateResult && $privateResult['ok']) {
            return $privateResult;
        } else if ($userScopeResult && $userScopeResult['ok']) {
            return $userScopeResult;
        } else {
            // Her iki istek de başarısız olduysa, hata mesajını döndür
            if ($menuResult && isset($menuResult['description'])) {
                return $menuResult;
            } else if ($privateResult && isset($privateResult['description'])) {
                return $privateResult;
            } else if ($userScopeResult && isset($userScopeResult['description'])) {
                return $userScopeResult;
            } else {
                return ['ok' => false, 'description' => 'Bilinmeyen hata'];
            }
        }
    }
    
    /**
     * Bot bilgilerini getir
     * 
     * @return array|false
     */
    public function botBilgileriniGetir() {
        return $this->request('getMe', []);
    }
    
    /**
     * Bot komutlarını getir
     * 
     * @return array|false
     */
    public function botKomutlariniGetir() {
        return $this->request('getMyCommands', []);
    }
    
    /**
     * Bot açıklamasını getir
     * 
     * @return array|false
     */
    public function botAciklamasiniGetir() {
        return $this->request('getMyDescription', []);
    }
    
    /**
     * Bot kısa açıklamasını getir
     * 
     * @return array|false
     */
    public function botKisaAciklamasiniGetir() {
        return $this->request('getMyShortDescription', []);
    }
    
    /**
     * Bot profil fotoğrafını güncelle
     * 
     * @param string $foto_yolu Fotoğraf dosyasının yolu
     * @return array|false
     */
    public function botProfilFotografiGuncelle($foto_yolu) {
        // Doğru API metodunu kullan ve hata ayıklama için detaylı log tut
        $params = [
            'photo' => new CURLFile($foto_yolu)
        ];
        
        // Önce uploadProfilePhoto metodunu dene
        $sonuc = $this->request('uploadProfilePhoto', $params);
        if ($sonuc && $sonuc['ok']) {
            return $sonuc;
        }
        
        // Eğer başarısız olursa, setProfilePhoto metodunu dene
        $sonuc = $this->request('setProfilePhoto', $params);
        if ($sonuc && $sonuc['ok']) {
            return $sonuc;
        }
        
        // Son çare olarak, setMyPhoto metodunu dene
        return $this->request('setMyPhoto', $params);
    }
    
    /**
     * Webhook bilgisini getir
     * 
     * @return array|false
     */
    public function webhookBilgisiniGetir() {
        return $this->request('getWebhookInfo', []);
    }
    
    /**
     * Webhook ayarla
     * 
     * @param string $url Webhook URL
     * @param string|null $certificate SSL sertifikası dosya yolu
     * @param int $max_connections Maksimum bağlantı sayısı
     * @param array $allowed_updates İzin verilen güncelleme türleri
     * @return array|false
     */
    public function webhookAyarla($url, $certificate = null, $max_connections = 40, $allowed_updates = ['message', 'callback_query']) {
        $params = [
            'url' => $url,
            'max_connections' => $max_connections,
            'allowed_updates' => json_encode($allowed_updates)
        ];
        
        if (!is_null($certificate)) {
            $params['certificate'] = new CURLFile($certificate);
        }
        
        return $this->request('setWebhook', $params);
    }
    
    /**
     * Webhook kaldır
     * 
     * @return array|false
     */
    public function webhookKaldir() {
        return $this->request('deleteWebhook', []);
    }
    
    /**
     * Telegram API'ye istek gönder
     * 
     * @param string $method API metodu
     * @param array $params Parametreler
     * @return array API yanıtı
     */
    public function request($method, $params = []) {
        $url = $this->apiUrl . $method;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        return json_decode($response, true);
    }
}
?>
